﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.MappingService.Helpers;
using VA.PPMS.ProviderData;
using Organization = VA.PPMS.ProviderData.Organization;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapProviderToCrm : MapperBase
    {
        private readonly MapCcnIdentifierToCrm _ccnIdentifierMapper;
        private readonly MapNpiToCrm _npiMapper;
        private readonly MapTinToCrm _tinMapper;
        private readonly MapOtherIdentifierToCrm _otherIdentifierMapper;
        private readonly MapProviderServiceToCrm _providerServiceMapper;
        private readonly MapBoardCertificationToCrm _boardCertificationMapper;
        private readonly MapDeaScheduleToCrm _deaScheduleMapper;
        private readonly MapLicensureToCrm _licensureMapper;
        private readonly MapAuthorizedOfficialToCrm _authorizedOfficialMapper;
        private readonly MapOtherNameToCrm _otherNameMapper;
        private readonly MapTaxonomyToCrm _taxonomyMapper;
        private readonly MapContactToCrm _contactMapper;
        private readonly MapProviderCredentialToCrm _providerCredentialMapper;
        private readonly MapCareSiteToCrm _careSiteMapper;

        public MapProviderToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper, MapCcnIdentifierToCrm ccnIdentifierMapper, MapNpiToCrm npiMapper, MapTinToCrm tinMapper,
            MapOtherIdentifierToCrm otherIdentifierMapper, MapProviderServiceToCrm providerServiceMapper, MapBoardCertificationToCrm boardCertificationMapper,
            MapDeaScheduleToCrm deaScheduleMapper, MapLicensureToCrm licensureMapper, MapAuthorizedOfficialToCrm authorizedOfficialMapper, MapOtherNameToCrm otherNameMapper,
            MapTaxonomyToCrm taxonomyMapper, MapContactToCrm contactMapper, MapProviderCredentialToCrm providerCredentialMapper, MapCareSiteToCrm careSiteMapper)
            : base(ppmsContextHelper, ppmsHelper)
        {
            _ccnIdentifierMapper = ccnIdentifierMapper;
            _npiMapper = npiMapper;
            _tinMapper = tinMapper;
            _otherIdentifierMapper = otherIdentifierMapper;
            _providerServiceMapper = providerServiceMapper;
            _boardCertificationMapper = boardCertificationMapper;
            _deaScheduleMapper = deaScheduleMapper;
            _licensureMapper = licensureMapper;
            _authorizedOfficialMapper = authorizedOfficialMapper;
            _otherNameMapper = otherNameMapper;
            _taxonomyMapper = taxonomyMapper;
            _contactMapper = contactMapper;
            _providerCredentialMapper = providerCredentialMapper;
            _careSiteMapper = careSiteMapper;
        }

        internal OptionSetValue RecordSource
        {
            get
            {
                switch (NetworkShorthand)
                {
                    case DasMessage.VaNetworkIdentifier:
                        return new OptionSetValue((int)Account_ppms_RecordSource.VAProvider);
                    case DasMessage.TriWestIdentifier:
                        return new OptionSetValue((int)Account_ppms_RecordSource.Triwest);
                    default:
                        return new OptionSetValue((int)Account_ppms_RecordSource.CCN);
                }
            }
        }

        public async Task<MapperResultDetail> Map(Provider provider, string networkId, bool forVaNetwork, Guid? owner = null)
        {
            NetworkShorthand = networkId;
            await SetNetworkByShorthand(networkId);
            ForVaNetwork = forVaNetwork;
            if (owner.HasValue)
                Owner = new EntityReference("team", owner.Value);
            else
                Owner = null;

            switch (provider.TransactionType)
            {
                case TransactionTypeList.Update: return await MapUpdate(provider);
                case TransactionTypeList.DeactivateProvider: return await MapDelete(provider, true);
                case TransactionTypeList.DeactivateRelationship: return await MapDelete(provider);
                default: return null;
            }
        }

        public async Task<MapperResultDetail> MapDelete(Provider provider, bool isFull = false)
        {
            var result = new MapperResultDetail();

            result.Action = MapperAction.Delete;
            if (isFull)
            {
                var requests = await DeleteProvider(provider);
                if (requests != null && requests.Any()) result.Requests = requests;
                else result.ValidationMessage = $"Deactivation failed, request could not be created. (Id: {provider.ProviderId})";
            }
            else
            {
                result.Requests = await DeleteChildren(provider);
            }

            if (!result.Requests.Any())
            {
                result.ValidationMessage = $"Deactivation failed, requests could not be created. (Id: {provider.ProviderId})";
            }
            else
            {
                using (var context = await PpmsContextHelper.GetOrganizationServiceProxyAsync())
                {
                    foreach (var item in result.Requests)
                    {
                        context.Execute(item);
                    }
                }
            }

            return result;
        }

        private async Task<MapperResultDetail> MapUpdate(Provider schemaProvider)
        {
            var result = new MapperResultDetail();

            try
            {
                result.SchemaId = schemaProvider.ProviderId;

                var crmEntity = await CheckForExistingProvider(schemaProvider);
                if (crmEntity == null)
                {
                    result.Action = MapperAction.Insert;

                    var validationMsgs = ValidationHelper.Validate(schemaProvider, ForVaNetwork);
                    var messages = validationMsgs as string[] ?? validationMsgs.ToArray();
                    if (messages.Any())
                    {
                        result.ValidationSource = "Provider";
                        result.ValidationMessage = ErrorHelper.FormatMessage(messages);
                    }
                    else
                    {
                        result.Provider = await MapInsert(schemaProvider);
                    }
                }
                else
                {
                    result.Action = MapperAction.Update;
                    result.Provider = await UpdateProvider(crmEntity, schemaProvider);
                }
            }
            catch (PpmsServiceException pse)
            {
                result.ValidationSource = pse.ErrorSource;
                result.ValidationMessage = $"Update failed: {pse.Message}";
            }

            return result;
        }

        private async Task<Account> MapInsert(Provider provider)
        {
            var crmProvider = new Account
            {
                Id = Guid.NewGuid(),
                ppms_isexternal = !ForVaNetwork,
                Fax = provider.Fax,
                ppms_providertype = ResolveProviderType(provider),
                Name = provider.ProviderNameDisplay,
                EMailAddress1 = provider.Email,
                Telephone1 = provider.Phone,
                ppms_RecordSource = RecordSource
            };

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) crmProvider.OwnerId = Owner;

            if (provider.Address != null)
            {
                crmProvider.Address1_Line1 = provider.Address.Address1;
                crmProvider.Address1_Line2 = provider.Address.Address2;
                crmProvider.Address1_Line3 = provider.Address.Address3;
                crmProvider.Address1_City = provider.Address.City;
                crmProvider.Address1_StateOrProvince = provider.Address.State;
                crmProvider.Address1_County = provider.Address.County;
                crmProvider.Address1_Country = provider.Address.CountryCode;
            }

            if (provider.HealthProviderTypeSpecified) crmProvider.ppms_externalhealthprovidertype = EnumHelper.MapEnumToOptionSetValue<Account_ppms_externalhealthprovidertype>(provider.HealthProviderType.ToString());
            if (!string.IsNullOrEmpty(provider.InternalType)) crmProvider.ppms_internaltype = EnumHelper.MapEnumToOptionSetValue<Account_ppms_internaltype>(provider.InternalType);

            // Set station number
            if (!string.IsNullOrEmpty(provider.StationNumber))
            {
                var facility = await PpmsHelper.GetFacilityByStationNumber(provider.StationNumber);
                if (facility != null) crmProvider.ppms_facilityid = facility;
            }

            if (provider.Licensures == null || provider.Licensures.Item.Any())
            {
                crmProvider.Attributes.Add("ppms_licensevalidated", true);
            }

            if (provider.IsOrganization)
            {
                var org = provider.Type.Item as Organization;

                if (org?.Locations != null) await InsertRelatedEntity(_careSiteMapper, org.Locations.Item, crmProvider);
                if (org?.AuthorizedOfficials != null) await InsertRelatedEntity(_authorizedOfficialMapper, org.AuthorizedOfficials.Item, crmProvider);
            }
            else if (provider.Type.Item is Individual individual)
            {
                crmProvider.ppms_religion = individual.Religion;

                // Set name values
                crmProvider.ppms_ProviderFirstName = individual.FirstName.Trim();
                crmProvider.ppms_ProviderLastName = individual.LastName.Trim();

                if (individual.EthnicitySpecified) crmProvider.ppms_ethnicity = EnumHelper.MapEnumToOptionSetValue<Account_ppms_ethnicity>(individual.Ethnicity.ToString());
                if (individual.GenderSpecified) crmProvider.ppms_gender = EnumHelper.MapEnumToOptionSetValue<ppms_Gender>(individual.Gender.ToString());
                if (individual.IsAcceptingNewPatientsSpecified) crmProvider.ppms_individualisacceptingnewpatients = individual.IsAcceptingNewPatients;
                if (individual.IsPrimaryCareProviderSpecified) crmProvider.ppms_PrimaryCarePhysician = individual.IsPrimaryCareProvider;
                if (individual.IsPrimaryCareProviderAcceptingVaSpecified) crmProvider.ppms_primarycareprovideracceptingva = individual.IsPrimaryCareProviderAcceptingVa;
                if (individual.ProviderCredentials != null) await InsertRelatedEntity(_providerCredentialMapper, individual.ProviderCredentials.Item, crmProvider);
                if (individual.ProviderServices != null)
                {
                    // Set target network
                    _providerServiceMapper.NetworkId = NetworkId;
                    await InsertRelatedEntity(_providerServiceMapper, individual.ProviderServices.Item, crmProvider);
                }
                if (individual.BoardCertifications != null) await InsertRelatedEntity(_boardCertificationMapper, individual.BoardCertifications.Item, crmProvider);
                if (individual.Contacts != null) await InsertRelatedEntity(_contactMapper, individual.Contacts.Item, crmProvider);
                if (individual.DeaNumbers != null) await InsertRelatedEntity(_deaScheduleMapper, individual.DeaNumbers.Item, crmProvider);
                if (individual.ProviderOtherNames != null) await InsertRelatedEntity(_otherNameMapper, individual.ProviderOtherNames.Item, crmProvider);
            }

            if (!string.IsNullOrEmpty(provider.ProviderId))
            {
                var id = new ProviderNetworkId
                {
                    Number = provider.ProviderId,
                    NetworkId = NetworkId.ToString()
                };
                // Set target network
                _ccnIdentifierMapper.NetworkId = NetworkId;
                // Map provider network
                var entity = await _ccnIdentifierMapper.MapInsert(id, crmProvider);
                crmProvider.ppms_account_providernetworkid = new List<ppms_providernetworkid> { entity };
            }

            var firstNpi = provider.FirstNpiNumber;
            if (!string.IsNullOrEmpty(firstNpi))
            {
                crmProvider.ppms_ProviderIdentifier = firstNpi;
                crmProvider.ppms_ProviderIdentifierType = new OptionSetValue((int)ppms_ProviderIdentifierType.NPI);
            }

            var idList = new List<ppms_provideridentifier>();
            // Do not include the first NPI, it will be processed as part of the provider identifier
            var npiList = GetNpiList(provider.Npis, firstNpi);
            if (npiList.Any())
            {
                var entity = await _npiMapper.InsertMapperEntity(npiList, crmProvider, true);
                idList.AddRange(ConvertEntityList<ppms_provideridentifier>(entity));
            }

            if (provider.Tins != null && provider.Tins.Item.Any())
            {
                var entity = await _tinMapper.InsertMapperEntity(provider.Tins.Item, crmProvider, true);
                idList.AddRange(ConvertEntityList<ppms_provideridentifier>(entity));
            }

            if (idList.Count > 0) crmProvider.ppms_account_ppms_provideridentifier_Provider = idList;
            if (provider.OtherIdentifiers != null) await InsertRelatedEntity(_otherIdentifierMapper, provider.OtherIdentifiers.Item, crmProvider);
            if (provider.Licensures != null) await InsertRelatedEntity(_licensureMapper, provider.Licensures.Item, crmProvider);
            if (provider.Specialties != null) await InsertRelatedEntity(_taxonomyMapper, provider.Specialties.Item, crmProvider);

            using (var context = await PpmsContextHelper.GetOrganizationServiceProxyAsync())
            {
                context.Create(crmProvider);
            }

            return crmProvider;
        }

        private IList<Npi> GetNpiList(Npis npis, string firstNpi)
        {
            if (npis != null && npis.Item.Any())
                return new List<Npi>();

            return npis.Item.Where(x => x.Number != firstNpi) as IList<Npi>;
        }

        private async Task<Account> UpdateProvider(Account crmEntity, Provider schemaProvider)
        {
            var crmProvider = new Account
            {
                Id = crmEntity.Id,
                Name = crmEntity.Name,
                //ppms_ProviderIdentifier = crmEntity.ppms_ProviderIdentifier,
                EMailAddress1 = crmEntity.EMailAddress1,
                Telephone1 = crmEntity.Telephone1,
                Fax = crmEntity.Fax,
                ppms_religion = crmEntity.ppms_religion,
                ppms_individualisacceptingnewpatients = crmEntity.ppms_individualisacceptingnewpatients,
                ppms_PrimaryCarePhysician = crmEntity.ppms_PrimaryCarePhysician,
                ppms_primarycareprovideracceptingva = crmEntity.ppms_primarycareprovideracceptingva,
                ppms_externalhealthprovidertype = crmEntity.ppms_externalhealthprovidertype,
                ppms_ethnicity = crmEntity.ppms_ethnicity,
                ppms_RecordSource = RecordSource
            };

            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                context.Attach(crmProvider);

                var newName = schemaProvider.ProviderNameDisplay;

                if (crmEntity.ppms_providertype.Value == (int)ppms_ProviderType.Individual) newName = schemaProvider.IndividualName;
                if (IsChanged(newName, crmProvider.Name)) crmProvider.Name = newName;

                // Update network relationships
                if (!crmEntity.ppms_account_providernetworkid.Any(x => x.ppms_providerid == schemaProvider.ProviderId.Trim()))
                {
                    var id = new ProviderNetworkId
                    {
                        Number = schemaProvider.ProviderId,
                        NetworkId = NetworkId.ToString()
                    };

                    await UpdateRelatedEntity(_ccnIdentifierMapper, new List<ProviderNetworkId> { id }, crmProvider, context);
                }

                if (IsChanged(schemaProvider.Email, crmProvider.EMailAddress1)) crmProvider.EMailAddress1 = schemaProvider.Email;
                if (IsChanged(schemaProvider.Phone, crmProvider.Telephone1)) crmProvider.Telephone1 = schemaProvider.Phone;

                if (schemaProvider.Address != null)
                {
                    if (IsChanged(schemaProvider.Address.Address1, crmProvider.Address1_Line1)) crmProvider.Address1_Line1 = schemaProvider.Address.Address1;
                    if (IsChanged(schemaProvider.Address.Address2, crmProvider.Address1_Line2)) crmProvider.Address1_Line2 = schemaProvider.Address.Address2;
                    if (IsChanged(schemaProvider.Address.Address3, crmProvider.Address1_Line3)) crmProvider.Address1_Line3 = schemaProvider.Address.Address3;
                    if (IsChanged(schemaProvider.Address.City, crmProvider.Address1_City)) crmProvider.Address1_City = schemaProvider.Address.City;
                    if (IsChanged(schemaProvider.Address.State, crmProvider.Address1_StateOrProvince)) crmProvider.Address1_StateOrProvince = schemaProvider.Address.State;
                    if (IsChanged(schemaProvider.Address.County, crmProvider.Address1_County)) crmProvider.Address1_County = schemaProvider.Address.County;
                    if (IsChanged(schemaProvider.Address.CountryCode, crmProvider.Address1_Country)) crmProvider.Address1_Country = schemaProvider.Address.CountryCode;
                }

                if (IsChanged(schemaProvider.Fax, crmProvider.Fax)) crmProvider.Fax = schemaProvider.Fax;

                if (schemaProvider.IsOrganization)
                {
                    var org = schemaProvider.Type.Item as Organization;

                    if (org?.Locations != null) await UpdateRelatedEntity(_careSiteMapper, org.Locations.Item, crmProvider, context);
                    if (org?.AuthorizedOfficials != null) await UpdateRelatedEntity(_authorizedOfficialMapper, org.AuthorizedOfficials.Item, crmProvider, context);
                }
                else
                {
                    var individual = schemaProvider.Type.Item as Individual;

                    if (individual != null && IsChanged(individual.Religion, crmProvider.ppms_religion)) crmProvider.ppms_religion = individual.Religion;
                    if (individual != null && individual.IsAcceptingNewPatientsSpecified) crmProvider.ppms_individualisacceptingnewpatients = individual.IsAcceptingNewPatients;
                    if (individual != null && individual.IsPrimaryCareProviderSpecified) crmProvider.ppms_PrimaryCarePhysician = individual.IsPrimaryCareProvider;
                    if (individual != null && individual.IsPrimaryCareProviderAcceptingVaSpecified) crmProvider.ppms_primarycareprovideracceptingva = individual.IsPrimaryCareProviderAcceptingVa;
                    if (individual != null && individual.EthnicitySpecified) crmProvider.ppms_ethnicity = EnumHelper.MapEnumToOptionSetValue<Account_ppms_ethnicity>(individual.Ethnicity.ToString());
                    if (individual?.ProviderServices != null) await UpdateRelatedEntity(_providerServiceMapper, individual.ProviderServices.Item, crmProvider, context);
                    if (individual?.BoardCertifications != null) await UpdateRelatedEntity(_boardCertificationMapper, individual.BoardCertifications.Item, crmProvider, context);
                    if (individual?.Contacts != null) await UpdateRelatedEntity(_contactMapper, individual.Contacts.Item, crmProvider, context);
                    if (individual?.DeaNumbers != null) await UpdateRelatedEntity(_deaScheduleMapper, individual.DeaNumbers.Item, crmProvider, context);
                    if (individual?.ProviderOtherNames != null) await UpdateRelatedEntity(_otherNameMapper, individual.ProviderOtherNames.Item, crmProvider, context);
                }

                if (schemaProvider.HealthProviderTypeSpecified) crmProvider.ppms_externalhealthprovidertype = EnumHelper.MapEnumToOptionSetValue<Account_ppms_externalhealthprovidertype>(schemaProvider.HealthProviderType.ToString());

                // Set station number
                if (!string.IsNullOrEmpty(schemaProvider.StationNumber))
                {
                    var facility = await PpmsHelper.GetFacilityByStationNumber(schemaProvider.StationNumber);
                    if (facility != null) crmProvider.ppms_facilityid = facility;
                }

                if (schemaProvider.Npis != null && schemaProvider.Npis.Item.Any())
                {
                    // only add new items
                    var newList = new List<Npi>();
                    foreach (var item in schemaProvider.Npis.Item)
                    {
                        if (!crmEntity.ppms_account_ppms_provideridentifier_Provider.Any(x => x.ppms_ProviderIdentifier == item.Number.Trim()))
                            newList.Add(item);
                    }

                    if (newList.Count > 0)
                        await UpdateRelatedEntity(_npiMapper, schemaProvider.Npis.Item, crmProvider, context);
                }

                if (schemaProvider.Tins != null && schemaProvider.Tins.Item.Any()) await UpdateRelatedEntity(_tinMapper, schemaProvider.Tins.Item, crmProvider, context);
                if (schemaProvider.OtherIdentifiers != null) await UpdateRelatedEntity(_otherIdentifierMapper, schemaProvider.OtherIdentifiers.Item, crmProvider, context);
                if (schemaProvider.Licensures != null) await UpdateRelatedEntity(_licensureMapper, schemaProvider.Licensures.Item, crmProvider, context);

                if (schemaProvider.Specialties != null)
                {
                    var newItems = MapTaxonomyToCrm.GetNewItems(schemaProvider.Specialties.Item, crmEntity);
                    if (newItems.Count > 0) await UpdateRelatedEntity(_taxonomyMapper, newItems, crmProvider, context);
                }

                // Set owner to CCN
                if (!ForVaNetwork && Owner != null) crmProvider.OwnerId = Owner;

                context.UpdateObject(crmProvider);
                context.SaveChanges();
            }

            return crmProvider;
        }

        private static OptionSetValue ResolveProviderType(Provider provider)
        {
            var typeValue = provider.IsOrganization ? (int)ppms_ProviderType.GroupPracticeAgency : (int)ppms_ProviderType.Individual;

            return new OptionSetValue(typeValue);
        }

        private async Task<Account> CheckForExistingProvider(Provider provider)
        {
            if (!string.IsNullOrEmpty(provider.CorrelationId))
            {
                return await GetProviderByCorrelationId(provider.CorrelationId, true) as Account;
            }

            if (provider.Npis == null || !provider.Npis.Item.Any()) return null;

            if (provider.Npis.Item.Count > 1)
            {
                foreach (var item in provider.Npis.Item)
                {
                    if (await GetProviderByNpi(item.Number) is Account entity)
                    {
                        return entity;
                    }
                }
            }
            else
            {
                return await GetProviderByNpi(provider.Npis.Item[0].Number, true) as Account;
            }
            
            // TODO - How to find existing providers without NPI??

            return null;
        }

        private async Task<IList<SetStateRequest>> DeleteProvider(Provider provider)
        {
            var requests = new List<SetStateRequest>();

            // Retrieve provider
            Account entity = string.IsNullOrEmpty(provider.CorrelationId)
                ? await GetProviderByNpi(provider.FirstNpi.Number, true) as Account
                : await GetProviderByCorrelationId(provider.CorrelationId, true) as Account;

            // Make sure provider was found
            if (entity == null) throw new PpmsServiceException("Deactivation failed: Provider does not exist.");

            var services = entity.ppms_account_ppms_providerservice;

            if (services == null) return requests;

            var match = services.Where(p => p.ppms_network.Id != NetworkId);

            if (match.Any()) requests.AddRange(_providerServiceMapper.EntityDelete(entity.ppms_account_ppms_providerservice));
            else requests.Add(EntityDelete(entity));
            
            return requests;
        }

        private async Task<IList<SetStateRequest>> DeleteChildren(Provider provider)
        {
            var requests = new List<SetStateRequest>();

            // Retrieve provider
            Account entity = string.IsNullOrEmpty(provider.CorrelationId)
                ? await GetProviderByNpi(provider.FirstNpi.Number, true) as Account
                : await GetProviderByCorrelationId(provider.CorrelationId, true) as Account;

            if (entity == null) throw new PpmsServiceException("Deactivation failed: Provider does not exist.");

            if (provider.OtherIdentifiers != null) requests.AddRange(DeleteRelatedEntity(_otherIdentifierMapper, provider.OtherIdentifiers.Item, entity));
            if (provider.Npis != null) requests.AddRange(DeleteRelatedEntity(_npiMapper, provider.Npis.Item, entity));
            if (provider.Tins != null) requests.AddRange(DeleteRelatedEntity(_tinMapper, provider.Tins.Item, entity));
            if (provider.Licensures != null) requests.AddRange(DeleteRelatedEntity(_licensureMapper, provider.Licensures.Item, entity));
            if (provider.Specialties != null) requests.AddRange(DeleteRelatedEntity(_taxonomyMapper, provider.Specialties.Item, entity));

            if (provider.IsOrganization)
            {
                if (provider.Type.Item is Organization org && org.AuthorizedOfficials != null) requests.AddRange(DeleteRelatedEntity(_authorizedOfficialMapper, org.AuthorizedOfficials.Item, entity));
            }
            else
            {
                var individual = provider.Type.Item as Individual;

                if (individual?.BoardCertifications != null) requests.AddRange(DeleteRelatedEntity(_boardCertificationMapper, individual.BoardCertifications.Item, entity));
                if (individual?.Contacts != null) requests.AddRange(DeleteRelatedEntity(_contactMapper, individual.Contacts.Item, entity));
                if (individual?.ProviderServices != null) requests.AddRange(DeleteRelatedEntity(_providerServiceMapper, individual.ProviderServices.Item, entity));
                if (individual?.ProviderOtherNames != null) requests.AddRange(DeleteRelatedEntity(_otherNameMapper, individual.ProviderOtherNames.Item, entity));
                if (individual?.DeaNumbers != null) requests.AddRange(DeleteRelatedEntity(_deaScheduleMapper, individual.DeaNumbers.Item, entity));
            }

            return requests;
        }

        private async Task InsertRelatedEntity<T>(MapperRelatedBase mapper, IList<T> items, Entity parent)
        {
            if (items.Any())
            {
                if (Owner != null) mapper.Owner = Owner;
                mapper.References = References;
                await mapper.InsertMapperEntity(items, parent, false, ForVaNetwork);
            }
        }

        private async Task<IEnumerable<Entity>> UpdateRelatedEntity<T>(MapperRelatedBase mapper, IList<T> items, Entity parent, PpmsContext context = null)
        {
            if (items.Any())
            {
                mapper.SetContext(context);
                mapper.NetworkId = NetworkId;
                if (Owner != null) mapper.Owner = Owner;
                return await mapper.UpdateMapperEntity(items, parent, false, ForVaNetwork);
            }

            return null;
        }

        private IEnumerable<SetStateRequest> DeleteRelatedEntity<T>(MapperRelatedBase mapper, IList<T> items, Account parent)
        {
            return mapper.DeleteMapperEntity(items, parent);
        }
    }
}